/* 
 *      XAB 1.0
 *
 *		Adam Beguelin
 *      Carnegie Mellon University
 *      Pittsburgh Supercomputing Center
 *      26 Nov 1991
 *
 *      abmon.c
 *		
 *		Store xab events generated by PVM program to a file.
 *
 * This program stores the events generated by a PVM program.
 * The events can be displayed graphically or simply textually.
 * by using a seperate program (ie xab) or simply looking at the file.
 *
 */

/* Changes 
 * 05/07/92 Added size to XAB_snd,  adamb 
 * 05/11/92 Added vsnd/vrcv
 */

/*
  Each xab event must contain the following information:
	Proccess name and pid number of the process that generated
	the event and the event id.  ie, XAB_initsend is the event
	id for an initsend.
*/


#include <stdio.h>
#include <sys/param.h>
#include <sys/time.h>

#include "ab.h"

/*	Global variable for the output file */

FILE *about;


int 
ab_handle()
{
	static int no_enrolled = 0;
	int mlen, mtype, mpid;
	long mtime[2];
	struct timeval tv;
	struct timezone tz;
	char mcomp[MAXUPROCNAMELEN];
	int curevent, rc;

	int serial, type, len, instance, cnt;
	char component[MAXUPROCNAMELEN];

	mtype = rcv(-1);

	/* note the current time on the monitor's machine */
	
    if(gettimeofday(&tv, &tz)) {
        fprintf(stderr, "gettimeofday() choked, bailing\n");
        exit(-1);
        }

	/* if the message is weird just ignore it */
	if (mtype != XABMTYPE) {
		printf("Ignoring\n");
		return(0);
		}

	/* yank the time from the message */
	getnlong(mtime,2);

	/* Deal with the events */
	if(getnint(&curevent, 1)) {
		fprintf(stderr, "ab_handle: Getnint buffer empty!\n");
		return(0);
		}

	rcvinfo(&mlen, &mtype, mcomp, &mpid);
	
	fprintf(about, "%ld %ld %d %s %s %d", 
		mtime[0], mtime[1], curevent, deftostr(curevent), mcomp, mpid);
	switch(curevent) {

		case XAB_initsend:
			{ 
			int serial;

			getnint(&serial, 1);
			fprintf(about, "\t serial %d\n", serial);
			}
			break;

		case XAB_putnint:
		case XAB_putnshort:
		case XAB_putnlong:
		case XAB_putnfloat:
		case XAB_putndfloat:
		case XAB_putncplx:
		case XAB_putbytes:
		case XAB_putstring:
		case XAB_getnint:
		case XAB_getnshort:
		case XAB_getnlong:
		case XAB_getnfloat:
		case XAB_getndfloat:
		case XAB_getncplx:
		case XAB_getbytes:
		case XAB_getstring:
			{ 
			int rc, serial, count;

			getnint(&rc, 1);
			getnint(&serial, 1);
			getnint(&count, 1);
			fprintf(about, "\t rc %d, serial %d, count %d\n", rc, serial, count);
			}
			break;

		case XAB_barrier:
			{
			char name[MAXUPROCNAMELEN];
			int number;

			getstring(name);
			getnint(&number, 1);

			fprintf(about, "\t name %s, number %d\n", name, number);
			}
			break;

		case XAB_barrier_done:
			getnint(&rc,1);
			fprintf(about, "\t rc %d\n", rc);
			break;

		case XAB_enroll:
			{
			char host[MAXUPROCNAMELEN];

			++no_enrolled;
			/* | hostname | */
			getstring(host);
			fprintf(about, "\t %s\n", host);
			}
			break;

		case XAB_initiate:
		case XAB_initiateM:
			{
			char comp[MAXUPROCNAMELEN];
			char arch[MAXUPROCNAMELEN];

			getstring(comp);
			getstring(arch);
			fprintf(about, "\t comp %s arch/host %s\n", comp, arch);
			}
			break;

		case XAB_initiate_done:
		case XAB_initiateM_done:
			getnint(&rc,1);
			fprintf(about, "\t pid %d\n", rc);
			break;

		case XAB_leave:
			--no_enrolled;
			fprintf(about, "\n");
			if (no_enrolled == 0) {
				/* print the local monitor's time for the event */
				fprintf(about, "\t %ld %ld\n", tv.tv_sec, tv.tv_usec);
				return(1);
				}
			break;

		case XAB_probe:
			getnint(&rc,1);
			fprintf(about, "\t type %d\n", rc);
			break;

		case XAB_pstatus:
			{
			int tmp[3];
			getnint(tmp, 3);
			fprintf(about, "\t rc %d, nproc %d, mixed %d\n", 
				tmp[0], tmp[1], tmp[2]);
			}
			break;

		case XAB_vrcv:
		case XAB_rcv:
			getnint(&type, 1);
			fprintf(about,  "\t type %d\n", type);
			break;

		case XAB_rcv_multi:
			{ 
			int ntypes, *types;

			getnint(&ntypes, 1);
			types = (int *)malloc(ntypes*sizeof(int));
			if (!types) {
				fprintf(stderr, "Yikes!\n");
				exit(-1);
				}
			getnint(types, ntypes);

			/* print out those types */
			fprintf(about, "\t");
			while(--ntypes) {
				fprintf(about, " %d", *types++);
				}
			fprintf(about, "\n");
			}
			break;

		case XAB_rcv_done:
		case XAB_vrcv_done:
		case XAB_rcv_multi_done:
			{
			int rc, spid, mtype, mlen, mser;
			char scomp[MAXUPROCNAMELEN];

			getnint(&rc, 1);
			getstring(scomp);
			getnint(&mpid, 1);
			getnint(&mtype, 1);
			getnint(&mlen, 1);
			getnint(&mser, 1);
			fprintf(about,"\t rc %d comp %s pid %d type %d len %d ser %d\n",
				rc, scomp, mpid, mtype, mlen, mser);
			}
			break;

		case XAB_rcvinfo:
		case XAB_ready:
		case XAB_status:
			getnint(&rc, 1);
			fprintf(about, "\t %d\n", rc);
			break;

		case XAB_vsnd:
		case XAB_snd:
			{
			int rc;
			char dcomp[MAXUPROCNAMELEN];
			int dpid, mtype, mser, msize;

			getnint(&rc, 1);
			getstring(dcomp);
			getnint(&dpid, 1);
			getnint(&mtype, 1);
			getnint(&mser, 1);
			getnint(&msize, 1);

			fprintf(about,"\t rc %d dcomp %s dpid %d mtype %d len %d mser %d\n",
				rc, dcomp, dpid, mtype, msize, mser);

			}
			break;

		case XAB_terminate:
			{
			char comp[MAXUPROCNAMELEN];
			int tmp[2];

			getstring(comp);
			getnint(tmp, 2);
			fprintf(about, "\t rc %d, comp %s, pid %d\n", tmp[1], comp, tmp[0]);
			/* if the terminate succeeded */
			if (rc == 0) --no_enrolled;
			/* if there are no more processes left */
			if (no_enrolled == 0) {
				/* print the local monitor's time for the event */
				fprintf(about, "\t %ld %ld\n", tv.tv_sec, tv.tv_usec);
				return(1);
				}
			break;
			}
			break;

		case XAB_waituntil:
			{
			char event[MAXUPROCNAMELEN];

			getstring(event);
			fprintf(about, "\t event %s\n", event);
			}
			break;

		case XAB_waituntil_done:
		case XAB_whoami:
			{
			int rc;
			char host[MAXUPROCNAMELEN];

			/* only count real enrolls, this forces programs to enroll
			XXX
			++no_enrolled;
			*/

			/* | hostname | rc | */
			getstring(host);
			getnint(&rc, 1);
			fprintf(about, "\t %s, rc %d\n", host, rc);
			}
			break;

		default:
			printf("%d no such event\n", curevent);
			break;

		} /* switch(curevent) */

	/* print the local monitor's time for the event */
	fprintf(about, "\t %ld %ld\n", tv.tv_sec, tv.tv_usec);

	/* flush the record if it's going to stdout */	
	if (about == stdout) fflush(about);

	return(0);
}

void
main(argc, argv)
int argc;
char *argv[];
{
	int mypid;

	/* 
		Parse command line arguments 
	*/

	if (argc > 2) goto usage;

	/* open the output file */
	about = stdout;
	if (argc == 2) {
		/* open the output file */
		about = fopen(argv[1], "w");
		/* if fopen fails */
		if (!about) {
			perror(argv[0]);
			exit(-1);
			}
		}


	/* connect to PVM */
	mypid = enroll(XAB);
	/* if i'm not pid 0 then bail.  multiple xabs might make sense in
	   the future but not now.  */
	if (mypid != 0) {
		fprintf(stderr, "other %s are enrolled.  I'm pid %d, bailing\n",
			XAB, mypid);
		leave();
		exit(-1);
		}
	
	/* while there are processes, handle the incoming monitor messages */
	while(!ab_handle())
		;

	leave();

	/* if the outfile isn't standard out then close it */
	if (about != stdout)
		fclose(about);

	/* see ya later */
	exit(0);

usage:
	fprintf(stderr,"usage: %s <out_file>\n", argv[0]);
	exit(-1);
}
